<?php
/**
 * ProcessComment SpecialPage for ArticleComments extension
 *
 * @file
 * @ingroup Extensions
 */

class SpecialProcessComment extends SpecialPage {

	/**
	 * Initialize the special page.
	 */
	public function __construct() {
		// A special page should at least have a name.
		// We do this by calling the parent class (the SpecialPage class)
		// constructor method with the name as first and only parameter.
		parent::__construct( 'ProcessComment' ,'comment' );
	}
	
	/**
	* Special page for comment processing.
	*/
	public function execute($param) {
		global $wgParser, $wgUser, $wgContentLang, $wgContLang;
		
		// Get a handle on the page so we can write text to it later
		$out = $this->getOutput();
		
		// Check user has permissions to execute this page (if not then the function will exit here)
		$this->checkPermissions();

		$wcl = ($wgContentLang ? $wgContentLang : $wgContLang);
		
		
		$request = $this->getRequest();
		$this->setHeaders();
		
		$out->setPageTitle( $this->msg( 'processcommenttitle' ) );
		//$out->addWikiMsg( 'processcommentbody' );
		
		
		$titleKey = $request->getText('titleKey');
		//$out->addWikiText("TitleKey: $titleKey");
		
		$titleNS = intval($request->getText('titleNS'));
		//$out->addWikiText("TitleNS: $titleNS");
		
		$commenterName = $request->getText('commenterName');
		//$out->addWikiText("CommenterName: $commenterName");
		
		$commenterURL = $request->getText('commenterURL');
		//$out->addWikiText("CommenterURL: $commenterURL");
		
		$commenterTownCountry = $request->getText('commenterTownCountry');
		//$out->addWikiText("Commenter Town, Country: $commenterTownCountry");
		
		$comment = $request->getText('comment');
		//$out->addWikiText("Comment: $comment");
		
		// Perform validation checks on supplied fields
		$ac = 'article-comments-';
		$messages = array();
		
		// If no title
		if (!$titleKey) $messages[] = wfMessage(
			$ac.'invalid-field', wfMessage($ac.'title-field')->inContentLanguage()->text(), $titleKey
			)->inContentLanguage()->text();
		
		if (!$commenterName) $messages[] = 
			wfMessage(
				$ac.'required-field', 
				wfMessage($ac.'name-string')->inContentLanguage()->text()
				)->inContentLanguage()->text();
				
				
		if (!$comment) $messages[] = 
			wfMessage(
				$ac.'required-field', 
				wfMessage($ac.'comment-string')->inContentLanguage()->text()
				);
				
		if (!empty($messages)) {
			$out->setPageTitle(wfMessage($ac.'submission-failed')->inContentLanguage()->text());
			$wikiText = "<div class='errorbox'>";
			$wikiText .= wfMessage($ac.'failure-reasons')->inContentLanguage()->text()."\n\n";
			foreach ($messages as $message) {
			    $wikiText .= "* $message\n";
			}
			$out->addWikiText($wikiText . "</div>");
			return;
		}
		
		// Setup title and talkTitle object
		$title = Title::newFromDBkey($titleKey);
		$title->mNamespace = $titleNS - ($titleNS % 2);
		$article = new Article($title);
		
		$talkTitle = Title::newFromDBkey($titleKey);
		$talkTitle->mNamespace = $titleNS + 1 - ($titleNS % 2);
		$talkArticle = new WikiPage($talkTitle);
		         
		// Check whether user is blocked from editing the talk page
		if ($wgUser->isLoggedIn()){
			if ($wgUser->isBlockedFrom($talkTitle)) {
				$out->setPageTitle(wfMessage($ac.'submission-failed')->inContentLanguage()->text());
				$wikiText = "<div class='errorbox'>";
				$wikiText .= wfMessage($ac.'failure-reasons')->inContentLanguage()->text()."\n\n";
				$wikiText .= '* '.wfMessage($ac.'user-is-blocked', $talkTitle->getPrefixedText())->inContentLanguage()->text()."\n";
				$out->addWikiText($wikiText . "</div>");
				return;
			}
		}
		
		// Retrieve article content
		$articleContent = '';
		if ( $article->exists() ) {
			$articleContent = $article->getContent();
		}
		
		// Retrieve existing talk content
		$talkContent = '';
		if ( $talkTitle->exists() ) {
			$talkContent = $talkArticle->getText();
		}
		
		// Check if talk NS is in the Namespace display list
		// Note: if so, then there's no need to confirm that <comments /> appears in the article or talk page.
		global $wgArticleCommentsNSDisplayList;
		$skipCheck = (
				is_array($wgArticleCommentsNSDisplayList) ?
				in_array($talkTitle->getNamespace(),$wgArticleCommentsNSDisplayList):
				false
			     );
	 
		// Check whether the article or its talk page contains a <comments /> flag
		if (!$skipCheck &&
			preg_match('/<comments( +[^>]*)?\\/>/', $articleContent)===0 &&
			preg_match('/<comments( +[^>]*)?\\/>/', $talkContent)===0
		) {
			$wgOut->setPageTitle(wfMessage($ac.'submission-failed')->inContentLanguage()->text());
			$wgOut->addWikiText(
				"<div class='errorbox'>".
				wfMessage($ac.'no-comments', $title->getPrefixedText())->inContentLanguage()->text().
				"</div>"
			);
			return;
		}
		
		// Run spam checks
		$isspam = false;
		$isspam = SpecialProcessComment::spamCheck( $comment , $commenterName, $commenterURL, $isspam );
		
		// If it's spam - it's gone!
		if ($isspam) {
			$out->setPageTitle(wfMessage($ac.'submission-failed')->inContentLanguage()->text());
			$out->addWikiText(
			    "<div class='errorbox'>".
			    wfMessage($ac.'no-spam')->inContentLanguage()->text().
			    "</div>"
			);
			return;
		}
	    
		// Initialize the talk page's content.
		if ( $talkContent == '' ) {
			$talkContent = wfMessage($ac.'talk-page-starter', $title->getPrefixedText() )->inContentLanguage()->text();
		}
	    
		// Determine signature components
		$d = $wcl->timeanddate( date( 'YmdHis' ), false, false) . ' (' . date( 'T' ) . ')';
		//$d = date("l dS \of F Y h:i:s A e");
		if ($commenterURL && $commenterURL!='http://') $sigText = "[$commenterURL $commenterName]";
		else if ($wgUser->isLoggedIn()) $sigText = $wgParser->getUserSig( $wgUser );
		else $sigText = $commenterName;
		
		// add town and country to signature
		if ($commenterTownCountry) $sigText .= ", " . $commenterTownCountry;
	
		$sigText .= ", ";
		
		// Search for insertion point, or append most recent comment.
		$commentText = wfMessage(
			$ac.'new-comment',
			wfMessage($ac.'commenter-said', $commenterName)->inContentLanguage()->text(),
			$comment,
			$sigText,
			$d
			)->inContentLanguage()->text();
		$posAbove = stripos( $talkContent, '<!--COMMENTS_ABOVE-->' );
		if ($posAbove===false) $posBelow = stripos( $talkContent, '<!--COMMENTS_BELOW-->' );
		if ($posAbove!==false) {
			// Insert comments above HTML marker
			$talkContent = substr( $talkContent, 0, $posAbove ) . $commentText . substr( $talkContent, $posAbove );
		} else if ($posBelow!==false) {
			// Insert comments below HTML marker
			$talkContent = substr( $talkContent, 0, $posBelow + 21 ) . $commentText . substr( $talkContent, $posBelow + 21 );
		} else {
			// No marker found, append to bottom (default)
			$talkContent .= $commentText;
		}
		
		// Update the talkArticle with the new comment
		$summary = wfMessage($ac.'summary', $commenterName)->inContentLanguage()->text();
		if (method_exists($talkArticle, 'doEdit')) {
			$talkArticle->doEdit($talkContent, $summary);
		} else {
			$method = ($talkArticle->exists() ? 'updateArticle' : 'insertNewArticle' );
			$talkArticle->$method($talkContent, $summary, false, false);
			return true;
		}

		$out->setPageTitle(wfMessage($ac.'submission-succeeded')->inContentLanguage()->text());
		$out->addWikiText(wfMessage($ac.'submission-success', $title->getPrefixedText())->inContentLanguage()->text());
		$out->addWikiText(wfMessage($ac.'submission-view-all', $talkTitle->getPrefixedText())->inContentLanguage()->text());
		
		return true;
	}
	
	/**
	* Checks ArticleComment fields for SPAM.
	* Usage: $wgHooks['ArticleCommentsSpamCheck'][] = 'defaultArticleCommentSpamCheck';
	* @param String $comment The comment body submitted (passed by value)
	* @param String $commenterName Name of commenter (passed by value)
	* @param String $commenterURL Website URL provided for comment (passed by value)
	* @param Boolean $isspam Whether the comment is spam (passed by reference)
	* @return Boolean Always true to indicate other hooking methods may continue to check for spam.
	*/
	private static function spamCheck($comment, $commenterName, $commenterURL, &$isspam) {
	
		# Short-circuit if spam has already been determined
		if ($isspam) return true;
		$fields = array($comment, $commenterName, $commenterURL);
		
		# Run everything through $wgSpamRegex if it has been specified
		global $wgSpamRegex;
		if ($wgSpamRegex) {
			foreach ($fields as $field) {
				if ( preg_match( $wgSpamRegex, $field ) ) return $isspam = true;
			}
		}
	
		# Rudimentary spam protection
		$spampatterns = array(
			'%\\[url=(https?|ftp)://%smi',
			'%<a\\s+[^>]*href\\s*=\\s*[\'"]?\\s*(https?|ftp)://%smi'
		);
		foreach ($spampatterns as $sp) {
			foreach (array($comment, $commenterName, $commenterURL) as $field) {
				if ( preg_match($sp, $field) ) return $isspam = true;
			}
		}
		
		# Check for bad input for commenterName (seems to be a popular spam location)
		$spampatterns = array(
			'%<a\\s+%smi',
			'%(https?|ftp)://%smi',
			'%(\\n|\\r)%smi'
		);
		foreach ($spampatterns as $sp) if ( preg_match($sp, $commenterName) ) return $isspam = true;
		
		# Fail for length violations
		if ( strlen($commenterName)>255 || strlen($commenterURL)>300 ) return $isspam = true;
		
		# We made it this far so it must not be spam
		return false;
	}
}